/**
 * Side Switcher
 * Version 0.0.5
 *
 * A UI element that is an alternative to the classic Toggle Button that allows users to pick from opposite/different states.
 * Unlike the classic Toggle, Side Switcher can have more than 2 options and is always attached to the edge of the screen,
 * allowing optimal use of the lens's free space without blocking the user's face.
 *
 * API:
 * switchToNext(): void - switches to the next state.
 * switchTo(icon: number): void - switches to the specified icon.
 * show(): void - animated shows side switcher.
 * hide(): void - animated hides side switcher.
 * disableIconBackground(): void - turns off the white icon's background.
 * enableIconBackground(): void - turns on the white icon's background.
 * enableInteractable(): void - allows interaction with the side switcher.
 * disableInteractable(): void - makes side switcher non-interactable.
 * (read-only) tooltip?: Tooltip - tooltip's script component if enabled.
 * (read-only) visible: boolean - whether the UI element is currently present.
 * activeIconIndex: number - index of the current active (selected) icon.
 * iconOpacity: number - an opacity of the icon.
 * backgroundOpacity: number - an opacity of the background.
 * icons: Texture[] - an array of switching icons.
 *
 * Events:
 *
 * onSwitch - triggers when switching (by touch and by API).
 * onManuallySwitch - triggers when switching by API.
 * onTouchStart - triggers when the touch starts.
 * onTouchMove - triggers when the touch moves.
 * onTouchEnd - triggers when the touch ends.
 * onEnabledInteractable - triggers when interactable enables.
 * onDisabledInteractable - triggers when interactable disables.
 * onShow - triggers when appearance animation starts.
 * onHide - triggers when disappearance animation starts.
 */
import { Event } from './Modules/Event/Event';
import { DestructionHelper } from './Modules/Scene/DestructionHelper';
import { PropertyAnimator } from './Modules/Animation/PropertyAnimator';
import { PassHelper } from './Modules/Scene/PassHelper';
import { ComponentWithDebug } from './Modules/Debug/ComponentWithDebug';
import { BehaviorEventCallbacks, CallbackType } from './Modules/BehaviorSupport/BehaviorEventCallbacks';
import {
    OffsetConfig
} from './Modules/Animation/Configs/OffsetConfig';
import {
    ScaleConfig
} from './Modules/Animation/Configs/ScaleConfig';
import { SceneHelper } from './Modules/Scene/SceneHelper';

enum Alignment {
    Left = -1,
    Right = 1
}

enum ShowOption {
    Auto,
    Delay,
    Custom
}

interface Tooltip extends BaseScriptComponent {
    autostart: boolean,
    label: string,
    show(): void,
    hide(): void,
    direction: "Left"|"Right"|"Top"|"Bottom",
}

@component
export class SideSwitcher extends ComponentWithDebug {

    @input
    private interactable: boolean;

    @ui.separator

    @input('int', '1')
    @widget(new ComboBoxWidget()
        .addItem('Left', -1)
        .addItem('Right', 1))
    private alignment: Alignment;

    @input
    @label('Icons')
    private _icons: Texture[];

    @ui.separator

    @input
    private startAnimation: boolean = true;

    @ui.separator

    @input
    @label('Tooltip')
    private tooltipEnabled: boolean;

    @ui.group_start('Tooltip')
    @showIf('tooltipEnabled')

    @input
    @label('Label')
    private tooltipLabel: string;

    @input('int', '0')
    @widget(new ComboBoxWidget()
        .addItem('Auto', 0)
        .addItem('Delay', 1)
        .addItem('Custom (API)', 2))
    @label('Show')
    private tooltipShowOption: ShowOption;

    @input
    @showIf('tooltipShowOption', 1)
    @label('Delay')
    private tooltipDelay: number;

    @ui.group_end

    @ui.separator

    @input('int')
    private renderOrder: number;

    @ui.separator

    @input
    private eventCallbacks: boolean;

    @ui.group_start('Event Callbacks')
    @showIf('eventCallbacks')

    @input('int', '0')
    @widget(new ComboBoxWidget()
        .addItem('None', 0)
        .addItem('Behavior Script', 1)
        .addItem('Behavior Custom', 2)
        .addItem('Custom Function', 3))
        callbackType: CallbackType;

    @input
    @showIf('callbackType', 1)
    private onSwitchBehaviors: ScriptComponent[];

    @input
    @showIf('callbackType', 2)
    private onSwitchCustomTriggers: string[];

    @input
    @showIf('callbackType', 3)
    @allowUndefined
    private customFunctionScript: ScriptComponent;

    @ui.separator
    @showIf('callbackType', 3)

    @input
    @showIf('callbackType', 3)
    private onSwitchFunctions: string[];

    @ui.group_end

    @input
    private prefabWorld: ObjectPrefab;

    @input
    private prefabPoints: ObjectPrefab;

    @typename
    private tooltipComponent;

    onSwitch: Event<number> = new Event();
    onManuallySwitch: Event<number> = new Event();
    onTouchStart: Event = new Event();
    onTouchMove: Event = new Event();
    onTouchEnd: Event = new Event();
    onEnabledInteractable: Event = new Event();
    onDisabledInteractable: Event = new Event();
    onShow: Event = new Event();
    onHide: Event = new Event();

    private readonly ANIMATION_DURATION_SHORT: number = 0.05;
    private readonly ANIMATION_DURATION_MEDIUM: number = 0.1;
    private readonly ANIMATION_DURATION_LONG: number = 0.15;

    private readonly ANIMATED_TOOLTIP_SCALE: vec3 = new vec3(1.3, 1.0, 1.0);
    private readonly DEFAULT_TOOLTIP_SCALE: vec3 = vec3.one();
    private readonly ANIMATED_POSITION_OFFSET: number = 0.3;

    private readonly POINTS_POSITION_OFFSET: number = 53;
    private readonly POINTS_POSITION_RATION: number = -0.25;

    private readonly helper: DestructionHelper = new DestructionHelper();

    private isVisible: boolean = true;
    private activeIcon: number = 0;

    private _iconOpacity: number = 1.0;
    private _backgroundOpacity: number = 1.0;
    private disabledBackgroundOpacity: number = 0.3;

    private startWorldPositionOffset: number = 1.0;
    private worldPositionRatio: number = 1.0;

    private thisSO: SceneObject;
    private parentSO: SceneObject;

    private interactionC: InteractionComponent;

    private backgroundSO: SceneObject;
    private backgroundST: ScreenTransform;
    private backgroundImage: Image;

    private iconBackgroundSO: SceneObject;
    private iconBackgroundImage: Image;

    private _tooltip: Tooltip;

    private iconSO: SceneObject;
    private iconImage: Image;

    private offsetAnimation: PropertyAnimator;
    private scaleAnimation: PropertyAnimator;
    private alphaAnimation: PropertyAnimator;

    private unitType: Canvas.UnitType;

    onAwake() {
        this.thisSO = this.getSceneObject();
        if (!this.validateCamera()) {
            return;
        }
        this.initialize();
    }

    /**
     * Returns whether the UI element is currently present.
     */
    get visible(): boolean {
        return this.isVisible;
    }

    /**
     * Getter for the tooltip's Script Component.
     */
    get tooltip(): Tooltip {
        return this._tooltip;
    }

    /**
     * Sets an icon's opacity.
     * @param value
     */
    set iconOpacity(value: number) {
        this._iconOpacity = value;
        this.setIconOpacity(value);
    }

    /**
     * Returns an icon's opacity.
     */
    get iconOpacity(): number {
        return this.iconImage.mainPass.baseColor.a;
    }

    /**
     * Sets a new background's opacity.
     * @param value
     */
    set backgroundOpacity(value: number) {
        if (this.interactable) {
            this._backgroundOpacity = value;
        } else {
            this.disabledBackgroundOpacity = value;
        }
        this.setBackgroundOpacity(value);
    }

    /**
     * Returns a background's opacity.
     */
    get backgroundOpacity(): number {
        return this.backgroundImage.mainPass.baseColor.a;
    }

    /**
     * Setter for an array of icons.
     * @param textures
     */
    set icons(textures: Texture[]) {
        this._icons = textures;
        this.updateIconTexture();
    }

    /**
     * Retrieves an array of the icons textures.
     */
    get icons(): Texture[] {
        return this._icons;
    }

    /**
     * Changes active(selected) icon without animation.
     * @param idx
     */
    set activeIconIndex(idx: number) {
        if (this.activeIcon == idx) return;
        this.activeIcon = idx;
        this.updateIconTexture();
        this.onManuallySwitch.trigger(this.activeIcon);
    }

    /**
     * Retrieves an index of the active(selected) icon.
     */
    get activeIconIndex(): number {
        return this.activeIcon;
    }

    /**
     * Switches to the next icon with animation.
     */
    switchToNext(): void {
        this.activeIcon = (this.activeIcon + 1) % this._icons.length;
        this.onManuallySwitch.trigger(this.activeIcon);
        this.switchAnimation();
    }

    /**
     * Switches to the specific icon with animation.
     * @param icon
     */
    switchTo(icon: number): void {
        if (this.activeIcon == icon) {
            return;
        }
        if (icon >= this._icons.length) {
            this.printWarning('trying to switch to an invalid icon');
        }
        this.activeIcon = icon;
        this.onManuallySwitch.trigger(this.activeIcon);
        this.switchAnimation();
    }

    /**
     * Animated shows side switcher.
     */
    show(): void {
        if (this.isVisible) {
            return;
        }
        this.isVisible = true;
        this.showAnimation();
        this.onShow.trigger();
    }

    /**
     * Animated hides side switcher.
     */
    hide(): void {
        if (!this.isVisible) {
            return;
        }
        this.isVisible = false;
        this.hideAnimation();
        this.onHide.trigger();
    }

    /**
     * Disables the white icon's background.
     */
    disableIconBackground(): void {
        this.iconBackgroundImage.enabled = false;
    }

    /**
     * Enables the white icon's background.
     */
    enableIconBackground(): void {
        this.iconBackgroundImage.enabled = true;
    }

    /**
     * Enables interactable and updates background opacity to "on" state.
     */
    enableInteractable(): void {
        if (this.interactable) {
            return;
        }
        this.interactable = true;
        this.setBackgroundOpacity(this._backgroundOpacity);
        this.onEnabledInteractable.trigger();
    }

    /**
     * Disables interactable and updates background opacity to "off" state.
     */
    disableInteractable(): void {
        if (!this.interactable) {
            return;
        }
        this.interactable = false;
        this.setBackgroundOpacity(this.disabledBackgroundOpacity);
        this.onDisabledInteractable.trigger();
    }

    /**
     * Initializes all components of the side switcher.
     * @private
     */
    private initialize(): void {
        this.initializeST();
        this.initializePrefab();
        this.initializeAlignment();
        this.initializeBackground();
        this.initializeIcon();
        this.initializeInteraction();
        this.initializeInteractable();
        this.initializeAnimations();
        this.showStartAnimation();
        this.initializeDestroyEv();
        this.initializeEventCallbacks();
        SceneHelper.setRenderLayerRecursively(this.thisSO, this.thisSO.layer);
        SceneHelper.setRenderOrderRecursively(this.thisSO, this.renderOrder);
        this.initializeTooltip();
    }

    private get alignmentOffset(): number {
        if (this.unitType === Canvas.UnitType.World) {
            return this.alignment;
        } else {
            return this.alignment * this.POINTS_POSITION_OFFSET * 2;
        }
    }

    private get positionOffset(): number {
        if (this.unitType === Canvas.UnitType.World) {
            return this.startWorldPositionOffset;
        } else {
            return this.alignment * -this.POINTS_POSITION_OFFSET;
        }
    }

    private get scaledPositionOffset(): number {
        if (this.unitType === Canvas.UnitType.World) {
            return this.ANIMATED_POSITION_OFFSET * this.worldPositionRatio;
        } else {
            return this.alignment * this.POINTS_POSITION_OFFSET * this.POINTS_POSITION_RATION;
        }
    }

    private validateCamera(): boolean {
        const camera: Camera = SceneHelper.findComponentRecursively(this.thisSO, 'Camera');
        if (isNull(camera) || camera.type !== Camera.Type.Orthographic) {
            print('Warning! Place Side Switcher under the Orthographic Camera.');
            return false;
        }
        return true;
    }

    /**
     * Adds a ScreenTransform to the script's SceneObject if necessary.
     * @private
     */
    private initializeST(): void {
        this.helper.getOrAddComponent(this.thisSO, 'ScreenTransform');
    }

    /**
     * Instantiates side switcher's prefab.
     * @private
     */
    private initializePrefab(): void {
        const canvas: Canvas = SceneHelper.findComponentRecursively(this.thisSO, 'Canvas');
        if (!isNull(canvas) && canvas.unitType !== Canvas.UnitType.World) {
            this.unitType = Canvas.UnitType.Points;
            this.parentSO = this.helper.instantiatePrefab(this.prefabPoints, this.thisSO, 'SideSwitcherPrefab');
        } else {
            this.unitType = Canvas.UnitType.World;
            this.parentSO = this.helper.instantiatePrefab(this.prefabWorld, this.thisSO, 'SideSwitcherPrefab');
        }
    }

    /**
     * Set's selected alignment.
     * @private
     */
    private initializeAlignment(): void {
        const alignmentName = this.alignment == Alignment.Left ? 'Left' : 'Right';
        this.parentSO.children.forEach((alignmentType) => {
            alignmentType.enabled = alignmentType.name.indexOf(alignmentName) >= 0;
            if (alignmentType.enabled) {
                this.backgroundSO = alignmentType;
            }
        });
        if (this.alignment == Alignment.Right) {
            this.startWorldPositionOffset = -0.5;
            this.worldPositionRatio = -0.5;
        }
    }

    /**
     * Initializes side switcher's background.
     * @private
     */
    private initializeBackground(): void {
        this.backgroundST = this.helper.getOrAddComponent(this.backgroundSO, 'ScreenTransform');
        this.backgroundImage = this.helper.getOrAddComponent(this.backgroundSO, 'Image');
        PassHelper.cloneAndReplaceMaterial(this.backgroundImage);
    }

    /**
     * Initializes tooltip if selected.
     * @private
     */
    private initializeTooltip(): void {
        if (this.tooltipEnabled) {
            this._tooltip = this.helper.createComponent(this.backgroundSO, this.tooltipComponent);
            this._tooltip.direction = (this.alignment === Alignment.Left) ? "Right" : "Left";
            this._tooltip.label = this.tooltipLabel;
            this._tooltip.autostart = false;
            switch (this.tooltipShowOption) {
                case ShowOption.Auto:
                    this._tooltip.show();
                    break;
                case ShowOption.Delay:
                    const tooltipDelay = this.createEvent('DelayedCallbackEvent');
                    tooltipDelay.bind(() => this._tooltip.show());
                    tooltipDelay.reset(this.tooltipDelay);
                    break;
            }
        }
    }

    /**
     * Initializes side switcher's icon.
     * @private
     */
    private initializeIcon(): void {
        this.iconBackgroundSO = SceneHelper.findChildObjectWithNameRecursively(this.backgroundSO, 'Icon Background');

        if (!this.iconBackgroundSO) {
            throw new Error('Side Switcher: Invalid Icon Background SceneObject.');
        }

        this.helper.getOrAddComponent(this.iconBackgroundSO, 'ScreenTransform');
        this.iconBackgroundImage = this.helper.getOrAddComponent(this.iconBackgroundSO, 'Image');
        PassHelper.cloneAndReplaceMaterial(this.iconBackgroundImage);
        this.iconSO = SceneHelper.findChildObjectWithNameRecursively(this.iconBackgroundSO, 'Icon');

        if (!this.iconBackgroundSO) {
            throw new Error('Side Switcher: Invalid Icon SceneObject.');
        }

        this.helper.getOrAddComponent(this.iconSO, 'ScreenTransform');
        this.iconImage = this.helper.getOrAddComponent(this.iconSO, 'Image');
        PassHelper.cloneAndReplaceMaterial(this.iconImage);
        if (this.icons.length > 0) {
            this.updateIconTexture();
        }
    }

    /**
     * Initializes Interaction Component.
     * @private
     */
    private initializeInteraction(): void {
        this.interactionC = this.helper.getOrAddComponent(this.backgroundSO, 'InteractionComponent');
        this.interactionC.onTouchStart.add(() => this.onInteractionTouchStart());
        this.interactionC.onTouchMove.add(() => this.onInteractionTouchMove());
        this.interactionC.onTouchEnd.add(() => this.onInteractionTouchEnd());
    }

    /**
     * Initializes interactable and initial state.
     * @private
     */
    private initializeInteractable(): void {
        if (!this.interactable) {
            this.setBackgroundOpacity(this.disabledBackgroundOpacity);
        }
    }

    /**
     * Handles touch start. Switches the icon if interactable is on.
     * @private
     */
    private onInteractionTouchStart(): void {
        if (!this.interactable) {
            return;
        }
        this.activeIcon = (this.activeIcon + 1) % this._icons.length;
        this.onTouchStart.trigger();
        this.switchAnimation();
        this.printDebug('Switched! Active Icon: ' + this.activeIcon);
    }

    private onInteractionTouchMove(): void {
        if (!this.interactable) {
            return;
        }
        this.onTouchMove.trigger();
    }

    private onInteractionTouchEnd(): void {
        if (!this.interactable) {
            return;
        }
        this.onTouchEnd.trigger();
    }

    /**
     * Shows the initial animation if selected.
     * @private
     */
    private showStartAnimation(): void {
        if (!this.startAnimation) {
            return;
        }
        this.setIconOpacity(0.0);
        this.backgroundST.offsets.setCenter(new vec2 (this.alignmentOffset, 0.0));
        this.showAnimation();
    }

    /**
     * Initializes animation for offset, scale and alpha.
     * @private
     */
    private initializeAnimations(): void {
        this.offsetAnimation = new PropertyAnimator({
            scriptComponent: this,
            defaultDuration: this.ANIMATION_DURATION_LONG
        });

        this.scaleAnimation = new PropertyAnimator({
            scriptComponent: this,
            defaultDuration: this.ANIMATION_DURATION_LONG
        });

        this.alphaAnimation = new PropertyAnimator({
            scriptComponent: this,
            defaultDuration: this.ANIMATION_DURATION_LONG
        });
        this.alphaAnimation.setupAnimation(() => this.getIconOpacity(),
            (value: number) => this.setIconOpacity(value),
            MathUtils.lerp);

        this.offsetAnimation.configure(new OffsetConfig(), this.backgroundST);
        this.scaleAnimation.configure(new ScaleConfig(), this.backgroundST);
    }

    /**
     * Initializes on destroy event.
     * @private
     */
    private initializeDestroyEv(): void {
        this.createEvent('OnDestroyEvent').bind((e) => {
            this.alphaAnimation = null;
            this.scaleAnimation = null;
            this.offsetAnimation = null;
            this.helper.destroyObjects();
            this.removeEvent(e);
        });
    }

    private initializeEventCallbacks(): void {
        if (this.eventCallbacks && this.callbackType !== CallbackType.None) {
            this.onSwitch.add(BehaviorEventCallbacks.invokeCallbackFromInputs(this, 'onSwitch'));
        }
    }

    /**
     * Animates icon's switching.
     * @private
     */
    private switchAnimation(): void {
        this.onSwitch.trigger(this.activeIcon);
        this.alphaAnimation.startAnimation(0.0, this.ANIMATION_DURATION_MEDIUM);
        this.alphaAnimation.setCallbackOnFinish(() => {
            this.updateIconTexture();
            this.alphaAnimation.startAnimation(this._iconOpacity, this.ANIMATION_DURATION_MEDIUM);
            this.alphaAnimation.setCallbackOnFinish(null);
        });
    }

    /**
     * Shows an animation of the appearance.
     * @private
     */
    private showAnimation(): void {
        this.offsetAnimation.startAnimation(new vec2(this.positionOffset, 0.0), this.ANIMATION_DURATION_MEDIUM);
        this.offsetAnimation.setCallbackOnFinish(() => {
            this.scaleAnimation.startAnimation(this.ANIMATED_TOOLTIP_SCALE, this.ANIMATION_DURATION_SHORT);
            this.offsetAnimation.startAnimation(new vec2(this.positionOffset + this.scaledPositionOffset, 0.0), this.ANIMATION_DURATION_SHORT);
            this.alphaAnimation.setCallbackOnFinish(null);
            this.offsetAnimation.setCallbackOnFinish(() => {
                this.scaleAnimation.startAnimation(this.DEFAULT_TOOLTIP_SCALE, this.ANIMATION_DURATION_SHORT);
                this.offsetAnimation.startAnimation(new vec2(this.positionOffset, 0.0), this.ANIMATION_DURATION_SHORT);
                this.offsetAnimation.setCallbackOnFinish(() => {
                    this.alphaAnimation.startAnimation(this._iconOpacity, this.ANIMATION_DURATION_LONG);
                    this.alphaAnimation.setCallbackOnFinish(null);
                });
            });
        });
    }

    /**
     * Shows an animation of the disappearance.
     * @private
     */
    private hideAnimation(): void {
        this.offsetAnimation.startAnimation(new vec2(this.alignmentOffset, 0.0), this.ANIMATION_DURATION_MEDIUM);
        this.alphaAnimation.startAnimation(0.0, this.ANIMATION_DURATION_LONG);
        this.alphaAnimation.setCallbackOnFinish(null);
        this.offsetAnimation.setCallbackOnFinish(null);
        if (this.tooltipEnabled) {
            this._tooltip.hide();
        }
    }

    /**
     * Sets an opacity of icon and icon's background.
     * @param value
     * @private
     */
    private setIconOpacity(value: number): void {
        PassHelper.setBaseColorAlpha(this.iconBackgroundImage.mainPass, value);
        PassHelper.setBaseColorAlpha(this.iconImage.mainPass, value);
    }

    /**
     * Sets an opacity of side switcher's background.
     * @param value
     * @private
     */
    private setBackgroundOpacity(value: number): void {
        PassHelper.setBaseColorAlpha(this.backgroundImage.mainPass, value);
    }

    private getIconOpacity(): number {
        return this.iconImage.mainPass.baseColor.a;
    }

    /**
     * Updates an icon's texture depending on index of the active icon.
     * @private
     */
    private updateIconTexture(): void {
        if (!this._icons[this.activeIcon]) {
            this.printWarning('icon with ' + this.activeIcon + ' index is missing!');
            return;
        }
        this.iconImage.mainPass.baseTex = this._icons[this.activeIcon];
    }
}
